<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Annonce;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class AnnonceController extends Controller
{
    /**
     * Afficher la liste des annonces avec filtres
     */
    public function index(Request $request)
    {
        $query = Annonce::with(['user', 'photos'])->active();

        // Filtres
        if ($request->has('type_bien')) {
            $query->byTypeBien($request->type_bien);
        }

        if ($request->has('statut')) {
            $query->byStatut($request->statut);
        }

        if ($request->has('ville')) {
            $query->byVille($request->ville);
        }

        if ($request->has('prix_min') && $request->has('prix_max')) {
            $query->byPrix($request->prix_min, $request->prix_max);
        }

        if ($request->has('superficie_min') && $request->has('superficie_max')) {
            $query->whereBetween('superficie', [$request->superficie_min, $request->superficie_max]);
        }

        if ($request->has('nombre_pieces')) {
            $query->where('nombre_pieces', $request->nombre_pieces);
        }

        $annonces = $query->orderBy('created_at', 'desc')->paginate(15);

        return response()->json([
            'success' => true,
            'data' => $annonces
        ]);
    }

    /**
     * Créer une nouvelle annonce
     */
    public function store(Request $request)
    {
        Log::info('API ANNONCE store: incoming request', [
            'user_id' => optional($request->user())->id,
            'payload' => $request->except(['photos']),
            'photos_count' => is_array($request->file('photos')) ? count($request->file('photos')) : 0,
            'type_bien_id' => $request->input('type_bien_id'),
            'type_bien' => $request->input('type_bien'),
        ]);
        
        // Préparer les données pour la validation (décoder equipements si JSON)
        $dataForValidation = $request->all();
        if (isset($dataForValidation['equipements']) && is_string($dataForValidation['equipements'])) {
            $decoded = json_decode($dataForValidation['equipements'], true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                $dataForValidation['equipements'] = $decoded;
            } else {
                $dataForValidation['equipements'] = [];
            }
        }
        
        // Vérifier si c'est un terrain
        $isTerrain = false;
        if (!empty($dataForValidation['type_bien_id'])) {
            $tb = \App\Models\TypeBien::find($dataForValidation['type_bien_id']);
            if ($tb && strtolower($tb->name) === 'terrain') {
                $isTerrain = true;
            }
        }
        
        // Règles de validation de base
        $rules = [
            'titre' => 'required|string|max:255',
            'description' => 'required|string',
            'type_bien_id' => 'nullable|exists:type_biens,id',
            'statut' => 'required|in:vente,location',
            'ville' => 'required|string|max:255',
            'quartier' => 'required|string|max:255',
            'adresse' => 'required|string',
            'prix' => 'required|numeric|min:0',
            'superficie' => $isTerrain ? 'required|numeric|min:0' : 'nullable|numeric|min:0', // Superficie obligatoire pour terrain
            'nombre_pieces' => 'nullable|integer|min:0',
            'nombre_chambres' => 'nullable|integer|min:0',
            'nombre_salles_bain' => 'nullable|integer|min:0',
            'equipements' => 'nullable|array',
            'photos' => 'nullable|array|max:10',
            'photos.*' => 'image|mimes:jpeg,png,jpg,gif|max:2048',
            // Champs spécifiques aux terrains (utilise les champs généraux : ville, quartier, adresse, superficie)
            'situation_geographique' => 'nullable|string', // Plus obligatoire, on utilise ville/quartier/adresse
            'localisation_exacte' => 'nullable|string', // Plus obligatoire, on utilise adresse
            'superficie_terrain' => 'nullable|numeric|min:0', // Plus obligatoire, on utilise superficie
            'type_terrain' => $isTerrain ? 'required|in:urbain,agricole' : 'nullable|in:urbain,agricole',
            'statut_approbation' => $isTerrain ? 'required|in:approuvé,non_approuvé' : 'nullable|in:approuvé,non_approuvé',
            'titre_propriete' => $isTerrain ? 'required|in:attestation,ADU,ACD,certificat_foncier' : 'nullable|in:attestation,ADU,ACD,certificat_foncier',
        ];
        
        $validator = Validator::make($dataForValidation, $rules);

        if ($validator->fails()) {
            Log::warning('API ANNONCE store: validation failed', [
                'errors' => $validator->errors()->toArray(),
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Erreur de validation',
                'errors' => $validator->errors()
            ], 422);
        }

        // Utiliser les données déjà décodées pour la validation
        $data = $dataForValidation;
        $data['user_id'] = $request->user()->id;

        // Si type_bien_id fourni et type_bien absent, essayer de le déduire
        if (!empty($data['type_bien_id']) && empty($data['type_bien'])) {
            $tb = \App\Models\TypeBien::find($data['type_bien_id']);
            if ($tb) {
                $data['type_bien'] = $tb->name;
            }
        }

        $annonce = Annonce::create($data);
        Log::info('API ANNONCE store: created annonce', [
            'annonce_id' => $annonce->id,
            'type_bien' => $annonce->type_bien,
            'type_bien_id' => $annonce->type_bien_id,
        ]);

        // Upload des photos
        if ($request->hasFile('photos')) {
            foreach ($request->file('photos') as $index => $photo) {
                $path = $photo->store('annonces/' . $annonce->id, 'public');
                
                $annonce->photos()->create([
                    'chemin' => $path,
                    'nom_original' => $photo->getClientOriginalName(),
                    'taille' => $photo->getSize(),
                    'type_mime' => $photo->getMimeType(),
                    'ordre' => $index,
                    'est_principale' => $index === 0,
                ]);
            }
        }

        Log::info('API ANNONCE store: success', [
            'annonce_id' => $annonce->id,
            'photos' => $annonce->photos()->count(),
        ]);
        return response()->json([
            'success' => true,
            'message' => 'Annonce créée avec succès',
            'data' => $annonce->load(['photos', 'user'])
        ], 201);
    }

    /**
     * Afficher une annonce spécifique
     */
    public function show(string $id)
    {
        $annonce = Annonce::with(['user', 'photos'])
            ->where('id', $id)
            ->active()
            ->first();

        if (!$annonce) {
            return response()->json([
                'success' => false,
                'message' => 'Annonce non trouvée'
            ], 404);
        }

        // Incrémenter le nombre de vues
        $annonce->increment('nombre_vues');

        return response()->json([
            'success' => true,
            'data' => $annonce
        ]);
    }

    /**
     * Mettre à jour une annonce
     */
    public function update(Request $request, string $id)
    {
        $annonce = Annonce::findOrFail($id);

        // Vérifier que l'utilisateur est le propriétaire de l'annonce
        if ($annonce->user_id !== $request->user()->id) {
            return response()->json([
                'success' => false,
                'message' => 'Non autorisé'
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'titre' => 'sometimes|required|string|max:255',
            'description' => 'sometimes|required|string',
            'type_bien' => 'sometimes|required|in:appartement,maison,studio,villa,terrain,bureau,commerce',
            'type_bien_id' => 'sometimes|required|integer|exists:type_biens,id',
            'statut' => 'sometimes|required|in:vente,location',
            'ville' => 'sometimes|required|string|max:255',
            'quartier' => 'sometimes|required|string|max:255',
            'adresse' => 'sometimes|required|string',
            'prix' => 'sometimes|required|numeric|min:0',
            'superficie' => 'nullable|numeric|min:0',
            'nombre_pieces' => 'nullable|integer|min:0',
            'nombre_chambres' => 'nullable|integer|min:0',
            'nombre_salles_bain' => 'nullable|integer|min:0',
            'equipements' => 'nullable|array',
            'statut_annonce' => 'sometimes|in:active,inactive,vendue,louee',
            'photos' => 'nullable|array|max:10',
            'photos.*' => 'image|mimes:jpeg,png,jpg,gif|max:2048',
            'photos_to_delete' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Erreur de validation',
                'errors' => $validator->errors()
            ], 422);
        }

        // Préparer les données pour la mise à jour
        $data = $request->except(['photos', 'photos_to_delete']);
        
        // Décoder equipements si JSON
        if (isset($data['equipements']) && is_string($data['equipements'])) {
            $decoded = json_decode($data['equipements'], true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                $data['equipements'] = $decoded;
            }
        }

        $annonce->update($data);

        // Supprimer les photos demandées
        if ($request->has('photos_to_delete')) {
            $photosToDeleteIds = json_decode($request->input('photos_to_delete'), true);
            if (is_array($photosToDeleteIds)) {
                foreach ($photosToDeleteIds as $photoId) {
                    $photo = $annonce->photos()->find($photoId);
                    if ($photo) {
                        Storage::disk('public')->delete($photo->chemin);
                        $photo->delete();
                    }
                }
            }
        }

        // Upload des nouvelles photos
        if ($request->hasFile('photos')) {
            $maxOrder = $annonce->photos()->max('ordre') ?? 0;
            foreach ($request->file('photos') as $index => $photo) {
                $path = $photo->store('annonces/' . $annonce->id, 'public');
                
                // La première photo devient principale si aucune principale n'existe
                $isMain = $index === 0 && $annonce->photos()->where('est_principale', true)->count() === 0;
                
                $annonce->photos()->create([
                    'chemin' => $path,
                    'nom_original' => $photo->getClientOriginalName(),
                    'taille' => $photo->getSize(),
                    'type_mime' => $photo->getMimeType(),
                    'ordre' => $maxOrder + 1 + $index,
                    'est_principale' => $isMain,
                ]);
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Annonce mise à jour avec succès',
            'data' => $annonce->load(['photos', 'user'])
        ], 200);
    }

    /**
     * Supprimer une annonce
     */
    public function destroy(Request $request, string $id)
    {
        $annonce = Annonce::findOrFail($id);

        // Vérifier que l'utilisateur est le propriétaire de l'annonce
        if ($annonce->user_id !== $request->user()->id) {
            return response()->json([
                'success' => false,
                'message' => 'Non autorisé'
            ], 403);
        }

        // Supprimer les photos du stockage
        foreach ($annonce->photos as $photo) {
            Storage::disk('public')->delete($photo->chemin);
        }

        $annonce->delete();

        return response()->json([
            'success' => true,
            'message' => 'Annonce supprimée avec succès'
        ]);
    }

    /**
     * Obtenir les annonces de l'utilisateur connecté
     */
    public function myAnnonces(Request $request)
    {
        $annonces = $request->user()->annonces()
            ->with(['photos'])
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return response()->json([
            'success' => true,
            'data' => $annonces
        ]);
    }

    /**
     * Obtenir les statistiques d'une annonce
     */
    public function stats(Request $request, string $id)
    {
        $annonce = Annonce::findOrFail($id);

        // Vérifier que l'utilisateur est le propriétaire de l'annonce
        if ($annonce->user_id !== $request->user()->id) {
            return response()->json([
                'success' => false,
                'message' => 'Non autorisé'
            ], 403);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'nombre_vues' => $annonce->nombre_vues,
                'nombre_favoris' => $annonce->nombre_favoris,
                'nombre_contacts' => $annonce->nombre_contacts,
            ]
        ]);
    }
}
