<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class AuthController extends Controller
{
    /**
     * Inscription d'un nouvel utilisateur
     */
    public function register(Request $request)
    {
        Log::info('API AUTH register: incoming request', [
            'path' => $request->path(),
            'payload' => $request->except(['password', 'password_confirmation']),
            'ip' => $request->ip(),
            'ua' => $request->userAgent(),
        ]);
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'numero' => 'required|string|max:20|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'role_id' => 'required|integer|exists:roles,id',
        ]);

        if ($validator->fails()) {
            Log::warning('API AUTH register: validation failed', [
                'errors' => $validator->errors()->toArray(),
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Erreur de validation',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'numero' => $request->numero,
            'password' => Hash::make($request->password),
            'role_id' => $request->role_id,
        ]);

        $token = $user->createToken('auth_token')->plainTextToken;

        Log::info('API AUTH register: success', [
            'user_id' => $user->id,
            'role_id' => $user->role_id,
        ]);
        return response()->json([
            'success' => true,
            'message' => 'Utilisateur créé avec succès',
            'data' => [
                'user' => $user->load('role'),
                'token' => $token,
                'token_type' => 'Bearer'
            ]
        ], 201);
    }

    /**
     * Connexion d'un utilisateur (avec email OU numero)
     */
    public function login(Request $request)
    {
        Log::info('API AUTH login: incoming request', [
            'path' => $request->path(),
            'payload' => $request->except(['password']),
            'ip' => $request->ip(),
            'ua' => $request->userAgent(),
        ]);
        
        $validator = Validator::make($request->all(), [
            'identifier' => 'required|string', // email ou numero
            'password' => 'required|string',
        ]);

        if ($validator->fails()) {
            Log::warning('API AUTH login: validation failed', [
                'errors' => $validator->errors()->toArray(),
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Erreur de validation',
                'errors' => $validator->errors()
            ], 422);
        }

        $identifier = $request->identifier;
        $password = $request->password;

        // Déterminer si l'identifiant est un email ou un numero
        $isEmail = filter_var($identifier, FILTER_VALIDATE_EMAIL);
        
        // Chercher l'utilisateur par email ou numero
        $user = null;
        if ($isEmail) {
            $user = User::where('email', $identifier)->first();
        } else {
            $user = User::where('numero', $identifier)->first();
        }

        // Vérifier les identifiants
        if (!$user || !Hash::check($password, $user->password)) {
            Log::warning('API AUTH login: invalid credentials', [
                'identifier' => $identifier,
                'type' => $isEmail ? 'email' : 'numero',
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Identifiants invalides'
            ], 401);
        }
        
        if (!$user->is_active) {
            Log::notice('API AUTH login: inactive account', [
                'user_id' => $user->id,
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Compte désactivé'
            ], 403);
        }

        $token = $user->createToken('auth_token')->plainTextToken;

        Log::info('API AUTH login: success', [
            'user_id' => $user->id,
        ]);
        return response()->json([
            'success' => true,
            'message' => 'Connexion réussie',
            'data' => [
                'user' => $user->load('role'),
                'token' => $token,
                'token_type' => 'Bearer'
            ]
        ]);
    }

    /**
     * Déconnexion de l'utilisateur
     */
    public function logout(Request $request)
    {
        Log::info('API AUTH logout: incoming request', [
            'user_id' => optional($request->user())->id,
        ]);
        $request->user()->currentAccessToken()->delete();

        Log::info('API AUTH logout: success');
        return response()->json([
            'success' => true,
            'message' => 'Déconnexion réussie'
        ]);
    }

    /**
     * Obtenir les informations de l'utilisateur connecté
     */
    public function me(Request $request)
    {
        Log::info('API AUTH me: incoming request', [
            'user_id' => optional($request->user())->id,
        ]);
        return response()->json([
            'success' => true,
            'data' => [
                'user' => $request->user()->load('role')
            ]
        ]);
    }

    /**
     * Connexion via Google
     */
    public function loginWithGoogle(Request $request)
    {
        Log::info('API AUTH login-google: incoming request', [
            'path' => $request->path(),
            'ip' => $request->ip(),
            'ua' => $request->userAgent(),
        ]);

        $validator = Validator::make($request->all(), [
            'id_token' => 'required|string',
            'email' => 'required|string|email',
            'name' => 'required|string|max:255',
            'profile_photo' => 'nullable|url',
            'google_id' => 'required|string',
        ]);

        if ($validator->fails()) {
            Log::warning('API AUTH login-google: validation failed', [
                'errors' => $validator->errors()->toArray(),
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Erreur de validation',
                'errors' => $validator->errors()
            ], 422);
        }

        // Vérifier si l'utilisateur existe déjà avec cet email ou google_id
        $user = User::where(function ($query) use ($request) {
            $query->where('email', $request->email)
                  ->orWhere('google_id', $request->google_id);
        })->first();

        if ($user) {
            // Mettre à jour le google_id si nécessaire
            if (!$user->google_id) {
                $user->google_id = $request->google_id;
                if ($request->profile_photo && !$user->profile_photo) {
                    $user->profile_photo = $request->profile_photo;
                }
                $user->save();
            }

            if (!$user->is_active) {
                return response()->json([
                    'success' => false,
                    'message' => 'Compte désactivé'
                ], 403);
            }

            $token = $user->createToken('auth_token')->plainTextToken;

            Log::info('API AUTH login-google: success (existing user)', [
                'user_id' => $user->id,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Connexion réussie',
                'data' => [
                    'user' => $user->load('role'),
                    'token' => $token,
                    'token_type' => 'Bearer'
                ]
            ]);
        }

        // Créer un nouvel utilisateur
        // Par défaut, on assigne le rôle Acheteur/Locataire (role_id = 3)
        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'google_id' => $request->google_id,
            'profile_photo' => $request->profile_photo,
            'numero' => null, // L'utilisateur pourra ajouter son numéro plus tard
            'password' => Hash::make(uniqid()), // Générer un mot de passe aléatoire
            'role_id' => $request->input('role_id', 3), // Par défaut Acheteur/Locataire
        ]);

        $token = $user->createToken('auth_token')->plainTextToken;

        Log::info('API AUTH login-google: success (new user)', [
            'user_id' => $user->id,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Inscription et connexion réussies',
            'data' => [
                'user' => $user->load('role'),
                'token' => $token,
                'token_type' => 'Bearer'
            ]
        ], 201);
    }

    /**
     * Connexion via Facebook
     */
    public function loginWithFacebook(Request $request)
    {
        Log::info('API AUTH login-facebook: incoming request', [
            'path' => $request->path(),
            'ip' => $request->ip(),
            'ua' => $request->userAgent(),
        ]);

        $validator = Validator::make($request->all(), [
            'access_token' => 'required|string',
            'email' => 'nullable|string|email', // Email optionnel car certains comptes Facebook n'ont pas d'email
            'name' => 'required|string|max:255',
            'profile_photo' => 'nullable|url',
            'facebook_id' => 'required|string',
        ]);

        if ($validator->fails()) {
            Log::warning('API AUTH login-facebook: validation failed', [
                'errors' => $validator->errors()->toArray(),
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Erreur de validation',
                'errors' => $validator->errors()
            ], 422);
        }

        // Vérifier si l'utilisateur existe déjà avec cet email (si fourni) ou facebook_id
        $user = User::where(function ($query) use ($request) {
            if ($request->email) {
                $query->where('email', $request->email)
                      ->orWhere('facebook_id', $request->facebook_id);
            } else {
                $query->where('facebook_id', $request->facebook_id);
            }
        })->first();

        if ($user) {
            // Mettre à jour le facebook_id si nécessaire
            if (!$user->facebook_id) {
                $user->facebook_id = $request->facebook_id;
                if ($request->profile_photo && !$user->profile_photo) {
                    $user->profile_photo = $request->profile_photo;
                }
                $user->save();
            }

            if (!$user->is_active) {
                return response()->json([
                    'success' => false,
                    'message' => 'Compte désactivé'
                ], 403);
            }

            $token = $user->createToken('auth_token')->plainTextToken;

            Log::info('API AUTH login-facebook: success (existing user)', [
                'user_id' => $user->id,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Connexion réussie',
                'data' => [
                    'user' => $user->load('role'),
                    'token' => $token,
                    'token_type' => 'Bearer'
                ]
            ]);
        }

        // Créer un nouvel utilisateur
        // Note: Si l'email n'est pas fourni, on génère un email unique basé sur facebook_id
        $email = $request->email;
        if (!$email) {
            // Générer un email unique si Facebook ne fournit pas d'email
            $email = 'facebook_' . $request->facebook_id . '@facebook.temp';
        }
        
        $user = User::create([
            'name' => $request->name,
            'email' => $email,
            'facebook_id' => $request->facebook_id,
            'profile_photo' => $request->profile_photo,
            'numero' => null, // L'utilisateur pourra ajouter son numéro plus tard
            'password' => Hash::make(uniqid()), // Générer un mot de passe aléatoire
            'role_id' => $request->input('role_id', 3), // Par défaut Acheteur/Locataire
        ]);

        $token = $user->createToken('auth_token')->plainTextToken;

        Log::info('API AUTH login-facebook: success (new user)', [
            'user_id' => $user->id,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Inscription et connexion réussies',
            'data' => [
                'user' => $user->load('role'),
                'token' => $token,
                'token_type' => 'Bearer'
            ]
        ], 201);
    }

    /**
     * Connexion via numéro de téléphone (préparé pour OTP)
     * NOTE: Cette méthode utilise maintenant 'numero' au lieu de 'phone'
     */
    public function loginWithPhone(Request $request)
    {
        Log::info('API AUTH login-phone: incoming request', [
            'path' => $request->path(),
            'payload' => $request->except(['password']),
            'ip' => $request->ip(),
            'ua' => $request->userAgent(),
        ]);
        $validator = Validator::make($request->all(), [
            'numero' => 'required|string|max:20',
            'password' => 'required|string',
        ]);

        if ($validator->fails()) {
            Log::warning('API AUTH login-phone: validation failed', [
                'errors' => $validator->errors()->toArray(),
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Erreur de validation',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::where('numero', $request->numero)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            Log::warning('API AUTH login-phone: invalid credentials', [
                'numero' => $request->numero,
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Numéro de téléphone ou mot de passe invalide'
            ], 401);
        }

        if (!$user->is_active) {
            Log::notice('API AUTH login-phone: inactive account', [
                'user_id' => $user->id,
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Compte désactivé'
            ], 403);
        }

        $token = $user->createToken('auth_token')->plainTextToken;

        Log::info('API AUTH login-phone: success', [
            'user_id' => $user->id,
        ]);
        return response()->json([
            'success' => true,
            'message' => 'Connexion réussie',
            'data' => [
                'user' => $user->load('role'),
                'token' => $token,
                'token_type' => 'Bearer'
            ]
        ]);
    }
}
