<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Favori;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class FavoriController extends Controller
{
    /**
     * Ajouter une annonce aux favoris
     */
    public function store(Request $request)
    {
        Log::info('API Favoris: store() called', [
            'user_id' => optional($request->user())->id,
            'payload' => $request->all(),
        ]);

        $validator = Validator::make($request->all(), [
            'annonce_id' => 'required|integer|exists:annonces,id',
        ]);

        if ($validator->fails()) {
            Log::warning('API Favoris: store() validation failed', [
                'errors' => $validator->errors()->toArray(),
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Erreur de validation',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = $request->user();

        // Vérifier si le favori existe déjà
        $existingFavori = Favori::where('user_id', $user->id)
            ->where('annonce_id', $request->annonce_id)
            ->first();

        if ($existingFavori) {
            Log::info('API Favoris: store() existing favori', [
                'user_id' => $user->id,
                'favori_id' => $existingFavori->id,
            ]);
            return response()->json([
                'success' => true,
                'message' => 'Annonce déjà dans les favoris',
                'data' => $existingFavori->load(['annonce', 'user']),
            ], 200);
        }

        // Créer le favori
        $favori = Favori::create([
            'user_id' => $user->id,
            'annonce_id' => $request->annonce_id,
        ]);

        $favori->load(['annonce.photos', 'annonce.user']);

        Log::info('API Favoris: store() favori created', [
            'user_id' => $user->id,
            'favori_id' => $favori->id,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Annonce ajoutée aux favoris',
            'data' => $favori,
        ], 201);
    }

    /**
     * Retirer une annonce des favoris
     */
    public function destroy($id)
    {
        Log::info('API Favoris: destroy() called', [
            'favori_id' => $id,
            'user_id' => optional(auth()->user())->id,
        ]);

        $favori = Favori::find($id);

        if (!$favori) {
            Log::warning('API Favoris: destroy() favori not found', [
                'favori_id' => $id,
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Favori non trouvé',
            ], 404);
        }

        // Vérifier que l'utilisateur est propriétaire du favori
        if ($favori->user_id !== auth()->id()) {
            Log::warning('API Favoris: destroy() unauthorized', [
                'favori_id' => $favori->id,
                'favori_user_id' => $favori->user_id,
                'request_user_id' => auth()->id(),
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Non autorisé',
            ], 403);
        }

        $favori->delete();

        Log::info('API Favoris: destroy() success', [
            'favori_id' => $id,
            'user_id' => auth()->id(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Annonce retirée des favoris',
        ], 200);
    }

    /**
     * Obtenir les favoris de l'utilisateur connecté
     */
    public function myFavoris(Request $request)
    {
        $user = $request->user();
        Log::info('API Favoris: myFavoris() called', [
            'user_id' => optional($user)->id,
            'headers' => $request->headers->all(),
            'token_present' => $request->bearerToken() !== null,
        ]);
        $perPage = $request->input('per_page', 15);
        $page = $request->input('page', 1);

        \Log::info('API Favoris: Début de la requête', [
            'user_id' => $user->id,
            'page' => $page,
            'per_page' => $perPage,
        ]);

        // Optimiser la requête : charger seulement les favoris avec des annonces existantes
        $favoris = Favori::where('user_id', $user->id)
            ->whereHas('annonce') // Seulement les favoris avec des annonces existantes
            ->with([
                'annonce.user',
                'annonce.photos' => function ($query) {
                    $query->orderBy('ordre');
                }
            ])
            ->orderBy('created_at', 'desc')
            ->paginate($perPage, ['*'], 'page', $page);

        \Log::info('API Favoris: Requête terminée', [
            'count' => $favoris->count(),
            'total' => $favoris->total(),
        ]);

        return response()->json([
            'success' => true,
            'data' => $favoris->items(),
            'pagination' => [
                'current_page' => $favoris->currentPage(),
                'last_page' => $favoris->lastPage(),
                'per_page' => $favoris->perPage(),
                'total' => $favoris->total(),
            ],
        ], 200);
    }

    /**
     * Vérifier si une annonce est dans les favoris de l'utilisateur
     */
    public function checkFavori(Request $request, $annonceId)
    {
        $user = $request->user();
        Log::info('API Favoris: checkFavori() called', [
            'user_id' => optional($user)->id,
            'annonce_id' => $annonceId,
        ]);

        $favori = Favori::where('user_id', $user->id)
            ->where('annonce_id', $annonceId)
            ->first();

        Log::info('API Favoris: checkFavori() result', [
            'found' => $favori !== null,
            'favori_id' => $favori?->id,
            'user_id' => $user->id,
            'annonce_id' => $annonceId,
        ]);

        return response()->json([
            'success' => true,
            'is_favori' => $favori !== null,
            'favori_id' => $favori?->id,
        ], 200);
    }
}
