<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Message;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class MessageController extends Controller
{
    /**
     * Récupérer la liste des conversations de l'utilisateur connecté
     * Retourne la dernière conversation avec chaque utilisateur
     */
    public function myMessages(Request $request)
    {
        try {
            $userId = $request->user()->id;
            
            Log::info("Récupération des conversations pour l'utilisateur", ['user_id' => $userId]);

            // Récupérer toutes les conversations (derniers messages groupés par utilisateur)
            $conversations = Message::select('messages.*')
                ->where(function ($query) use ($userId) {
                    $query->where('expediteur_id', $userId)
                          ->orWhere('destinataire_id', $userId);
                })
                ->with(['expediteur', 'destinataire', 'annonce'])
                ->orderBy('created_at', 'desc')
                ->get()
                ->groupBy(function ($message) use ($userId) {
                    // Grouper par l'autre utilisateur dans la conversation
                    return $message->expediteur_id == $userId 
                        ? $message->destinataire_id 
                        : $message->expediteur_id;
                })
                ->map(function ($messages) use ($userId) {
                    $lastMessage = $messages->first();
                    $otherUserId = $lastMessage->expediteur_id == $userId 
                        ? $lastMessage->destinataire_id 
                        : $lastMessage->expediteur_id;
                    
                    // Compter les messages non lus
                    $unreadCount = Message::where('destinataire_id', $userId)
                        ->where('expediteur_id', $otherUserId)
                        ->where('lu', false)
                        ->count();

                    return [
                        'other_user' => $lastMessage->expediteur_id == $userId 
                            ? $lastMessage->destinataire 
                            : $lastMessage->expediteur,
                        'last_message' => $lastMessage,
                        'unread_count' => $unreadCount,
                        'annonce' => $lastMessage->annonce,
                    ];
                })
                ->values();

            Log::info("Conversations récupérées avec succès", ['count' => $conversations->count()]);

            return response()->json($conversations);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la récupération des conversations", [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json(['message' => 'Erreur lors de la récupération des conversations'], 500);
        }
    }

    /**
     * Récupérer tous les messages d'une conversation avec un utilisateur spécifique
     */
    public function getConversation(Request $request, $otherUserId)
    {
        try {
            $userId = $request->user()->id;
            
            Log::info("Récupération de la conversation", [
                'user_id' => $userId,
                'other_user_id' => $otherUserId
            ]);

            // Récupérer tous les messages échangés entre les deux utilisateurs
            $messages = Message::where(function ($query) use ($userId, $otherUserId) {
                    $query->where('expediteur_id', $userId)
                          ->where('destinataire_id', $otherUserId);
                })
                ->orWhere(function ($query) use ($userId, $otherUserId) {
                    $query->where('expediteur_id', $otherUserId)
                          ->where('destinataire_id', $userId);
                })
                ->with(['expediteur', 'destinataire', 'annonce'])
                ->orderBy('created_at', 'asc')
                ->get();

            // Marquer tous les messages reçus comme lus
            Message::where('expediteur_id', $otherUserId)
                ->where('destinataire_id', $userId)
                ->where('lu', false)
                ->update([
                    'lu' => true,
                    'lu_at' => now()
                ]);

            Log::info("Conversation récupérée avec succès", ['messages_count' => $messages->count()]);

            return response()->json($messages);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la récupération de la conversation", [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json(['message' => 'Erreur lors de la récupération de la conversation'], 500);
        }
    }

    /**
     * Envoyer un message
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'destinataire_id' => 'required|exists:users,id',
                'annonce_id' => 'required|exists:annonces,id',
                'contenu' => 'required|string|max:5000',
            ]);

            Log::info("Envoi d'un message", [
                'expediteur_id' => $request->user()->id,
                'destinataire_id' => $validated['destinataire_id'],
                'annonce_id' => $validated['annonce_id']
            ]);

            $message = Message::create([
                'expediteur_id' => $request->user()->id,
                'destinataire_id' => $validated['destinataire_id'],
                'annonce_id' => $validated['annonce_id'],
                'contenu' => $validated['contenu'],
                'lu' => false,
            ]);

            $message->load(['expediteur', 'destinataire', 'annonce']);

            Log::info("Message envoyé avec succès", ['message_id' => $message->id]);

            return response()->json($message, 201);
        } catch (ValidationException $e) {
            Log::warning("Validation échouée lors de l'envoi du message", [
                'errors' => $e->errors()
            ]);
            return response()->json(['message' => 'Données invalides', 'errors' => $e->errors()], 422);
        } catch (\Exception $e) {
            Log::error("Erreur lors de l'envoi du message", [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json(['message' => 'Erreur lors de l\'envoi du message'], 500);
        }
    }

    /**
     * Marquer un message comme lu
     */
    public function markAsRead(Request $request, $id)
    {
        try {
            $message = Message::findOrFail($id);
            
            // Vérifier que l'utilisateur est bien le destinataire
            if ($message->destinataire_id !== $request->user()->id) {
                return response()->json(['message' => 'Non autorisé'], 403);
            }

            $message->marquerCommeLu();

            Log::info("Message marqué comme lu", ['message_id' => $id]);

            return response()->json(['message' => 'Message marqué comme lu']);
        } catch (\Exception $e) {
            Log::error("Erreur lors du marquage du message comme lu", [
                'message_id' => $id,
                'error' => $e->getMessage()
            ]);
            return response()->json(['message' => 'Erreur lors du marquage du message'], 500);
        }
    }

    /**
     * Supprimer un message
     */
    public function destroy(Request $request, $id)
    {
        try {
            $message = Message::findOrFail($id);
            
            // Vérifier que l'utilisateur est l'expéditeur
            if ($message->expediteur_id !== $request->user()->id) {
                return response()->json(['message' => 'Non autorisé'], 403);
            }

            $message->delete();

            Log::info("Message supprimé", ['message_id' => $id]);

            return response()->json(['message' => 'Message supprimé avec succès']);
        } catch (\Exception $e) {
            Log::error("Erreur lors de la suppression du message", [
                'message_id' => $id,
                'error' => $e->getMessage()
            ]);
            return response()->json(['message' => 'Erreur lors de la suppression du message'], 500);
        }
    }

    /**
     * Récupérer le nombre total de messages non lus
     */
    public function unreadCount(Request $request)
    {
        try {
            $count = Message::where('destinataire_id', $request->user()->id)
                ->where('lu', false)
                ->count();

            return response()->json(['unread_count' => $count]);
        } catch (\Exception $e) {
            Log::error("Erreur lors du comptage des messages non lus", [
                'error' => $e->getMessage()
            ]);
            return response()->json(['message' => 'Erreur lors du comptage des messages'], 500);
        }
    }
}
