<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class UserController extends Controller
{
    /**
     * Obtenir le profil de l'utilisateur connecté
     */
    public function profile(Request $request)
    {
        $user = $request->user();
        $user->load('role');
        
        return response()->json([
            'success' => true,
            'data' => $user,
        ]);
    }

    /**
     * Mettre à jour le profil de l'utilisateur
     */
    public function updateProfile(Request $request)
    {
        Log::info('UserController::updateProfile called', [
            'method' => $request->method(),
            'user_id' => optional($request->user())->id,
            'user_exists' => $request->user() !== null,
            'is_authenticated' => $request->user() !== null,
            'bearer_token_present' => $request->bearerToken() !== null,
            'headers' => $request->headers->all(),
        ]);
        
        $user = $request->user();
        
        if (!$user) {
            Log::error('UserController::updateProfile - User is null!');
            return response()->json([
                'success' => false,
                'message' => 'Non authentifié',
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|required|string|max:255',
            'email' => 'sometimes|required|string|email|max:255|unique:users,email,' . $user->id,
            'numero' => 'sometimes|required|string|max:20|unique:users,numero,' . $user->id,
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Erreur de validation',
                'errors' => $validator->errors()
            ], 422);
        }

        $user->update($request->only(['name', 'email', 'numero']));
        $user->load('role');

        Log::info('UserController::updateProfile - Profil mis à jour avec succès', [
            'user_id' => $user->id,
            'updated_fields' => $request->only(['name', 'email', 'numero']),
        ]);

        $response = response()->json([
            'success' => true,
            'message' => 'Profil mis à jour avec succès',
            'data' => $user,
        ], 200);

        Log::info('UserController::updateProfile - Réponse créée', [
            'status_code' => $response->getStatusCode(),
            'has_success' => true,
        ]);

        return $response;
    }

    /**
     * Mettre à jour la photo de profil
     */
    public function updateProfilePhoto(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'photo' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Erreur de validation',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = $request->user();

        try {
            // Supprimer l'ancienne photo si elle existe
            if ($user->profile_photo) {
                $oldPhotoPath = str_replace('/storage/', '', $user->profile_photo);
                if (Storage::disk('public')->exists($oldPhotoPath)) {
                    Storage::disk('public')->delete($oldPhotoPath);
                }
            }

            // Stocker la nouvelle photo
            $path = $request->file('photo')->store('profile-photos', 'public');
            
            // Mettre à jour le champ profile_photo avec l'URL complète
            $user->profile_photo = '/storage/' . $path;
            $user->save();

            Log::info('Profile photo updated', [
                'user_id' => $user->id,
                'path' => $user->profile_photo,
            ]);

            $user->load('role');

            return response()->json([
                'success' => true,
                'message' => 'Photo de profil mise à jour avec succès',
                'data' => $user,
            ]);
        } catch (\Exception $e) {
            Log::error('Error updating profile photo', [
                'user_id' => $user->id,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Erreur lors de la mise à jour de la photo',
            ], 500);
        }
    }

    /**
     * Récupérer un utilisateur par son ID
     */
    public function getUserById(Request $request, $id)
    {
        try {
            $user = User::with('role')->findOrFail($id);

            return response()->json([
                'success' => true,
                'data' => $user,
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching user by ID', [
                'user_id' => $id,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Utilisateur non trouvé',
            ], 404);
        }
    }
}
