<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class FixAuthorizationHeader
{
    /**
     * Handle an incoming request.
     * 
     * Certains proxies (Apache, nginx) suppriment le header Authorization.
     * Ce middleware récupère le token depuis X-Authorization si Authorization n'est pas présent.
     */
    public function handle(Request $request, Closure $next)
    {
        $authHeader = $request->header('Authorization');
        $xAuthHeader = $request->header('X-Authorization');
        
        // Vérifier dans $_SERVER car Apache peut mettre le header dans une variable d'environnement
        $serverAuth = $_SERVER['HTTP_AUTHORIZATION'] ?? $_SERVER['REDIRECT_HTTP_AUTHORIZATION'] ?? null;
        $serverXAuth = $_SERVER['HTTP_X_AUTHORIZATION'] ?? $_SERVER['REDIRECT_HTTP_X_AUTHORIZATION'] ?? null;
        
        Log::info('FixAuthorizationHeader: Middleware exécuté', [
            'path' => $request->path(),
            'has_authorization' => $authHeader !== null,
            'has_x_authorization' => $xAuthHeader !== null,
            'server_http_authorization' => $serverAuth !== null,
            'server_http_x_authorization' => $serverXAuth !== null,
            'auth_preview' => $authHeader ? substr($authHeader, 0, 30) . '...' : null,
            'x_auth_preview' => $xAuthHeader ? substr($xAuthHeader, 0, 30) . '...' : null,
            'server_auth_preview' => $serverAuth ? substr($serverAuth, 0, 30) . '...' : null,
            'server_x_auth_preview' => $serverXAuth ? substr($serverXAuth, 0, 30) . '...' : null,
        ]);
        
        // Si Authorization n'est pas présent, essayer de le récupérer depuis $_SERVER
        if (!$authHeader && $serverAuth) {
            Log::info('FixAuthorizationHeader: Récupération Authorization depuis $_SERVER', [
                'server_auth_preview' => substr($serverAuth, 0, 30) . '...',
            ]);
            $request->headers->set('Authorization', $serverAuth);
        }
        // Si Authorization n'est pas présent mais X-Authorization l'est, le copier
        elseif (!$authHeader && $xAuthHeader) {
            Log::info('FixAuthorizationHeader: Copie X-Authorization vers Authorization', [
                'x_auth_preview' => substr($xAuthHeader, 0, 30) . '...',
            ]);
            $request->headers->set('Authorization', $xAuthHeader);
        }
        // Si X-Authorization est dans $_SERVER mais pas dans les headers
        elseif (!$authHeader && $serverXAuth) {
            Log::info('FixAuthorizationHeader: Récupération X-Authorization depuis $_SERVER', [
                'server_x_auth_preview' => substr($serverXAuth, 0, 30) . '...',
            ]);
            $request->headers->set('Authorization', $serverXAuth);
        }
        
        return $next($request);
    }
}

