<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class LogAllRequests
{
    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next)
    {
        // Logger TOUTES les requêtes API
        $allHeaders = $request->headers->all();
        $authHeader = $request->header('Authorization') ?? $request->header('authorization') ?? null;
        $xAuthHeader = $request->header('X-Authorization') ?? $request->header('x-authorization') ?? null;
        $bearerToken = $request->bearerToken();
        
        // Vérifier dans $_SERVER aussi (parfois les headers sont là mais pas dans $request->headers)
        $serverAuth = $_SERVER['HTTP_AUTHORIZATION'] ?? $_SERVER['HTTP_X_AUTHORIZATION'] ?? null;
        
        Log::info('=== INCOMING REQUEST ===', [
            'path' => $request->path(),
            'method' => $request->method(),
            'full_url' => $request->fullUrl(),
            'ip' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'has_bearer' => $bearerToken !== null,
            'bearer_token' => $bearerToken ? substr($bearerToken, 0, 30) . '...' : null,
            'auth_header_raw' => $authHeader,
            'auth_header_preview' => $authHeader ? substr($authHeader, 0, 50) . '...' : null,
            'x_auth_header' => $xAuthHeader,
            'x_auth_preview' => $xAuthHeader ? substr($xAuthHeader, 0, 50) . '...' : null,
            'server_http_authorization' => $serverAuth,
            'all_header_keys' => array_keys($allHeaders),
            'headers' => $allHeaders,
            'server_keys' => array_keys($request->server->all()),
            'http_headers_in_server' => array_filter($request->server->all(), function($key) {
                return strpos($key, 'HTTP_') === 0;
            }, ARRAY_FILTER_USE_KEY),
        ]);

        $response = $next($request);

        // Récupérer le contenu de la réponse pour le logger
        $responseContent = null;
        try {
            $responseContent = $response->getContent();
            $responseData = json_decode($responseContent, true);
        } catch (\Exception $e) {
            $responseData = null;
        }

        $statusCode = $response->getStatusCode();
        
        // Log spécial pour les codes d'erreur avec success: true (anomalie)
        if ($statusCode >= 400 && isset($responseData['success']) && $responseData['success'] === true) {
            Log::warning('=== ANOMALY: Error status with success=true ===', [
                'path' => $request->path(),
                'method' => $request->method(),
                'status_code' => $statusCode,
                'response_success' => $responseData['success'],
                'response_message' => $responseData['message'] ?? null,
                'response_full' => $responseData,
            ]);
        }
        
        Log::info('=== RESPONSE ===', [
            'path' => $request->path(),
            'method' => $request->method(),
            'status' => $statusCode,
            'response_success' => $responseData['success'] ?? null,
            'response_message' => $responseData['message'] ?? null,
            'response_preview' => $responseContent ? substr($responseContent, 0, 200) . '...' : null,
        ]);

        return $response;
    }
}

