<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Laravel\Sanctum\PersonalAccessToken;

class LogSanctumToken
{
    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next)
    {
        $bearer = $request->bearerToken();
        $hashed = $bearer ? hash('sha256', $bearer) : null;
        
        // Chercher le token dans la base
        $token = null;
        if ($hashed) {
            $token = PersonalAccessToken::query()->where('token', $hashed)->first();
            
            // Si pas trouvé, chercher tous les tokens de la table pour debug
            if (!$token) {
                $allTokens = PersonalAccessToken::query()
                    ->whereNotNull('token')
                    ->orderBy('created_at', 'desc')
                    ->limit(5)
                    ->get(['id', 'tokenable_id', 'token', 'created_at']);
                
                Log::warning('Sanctum token NOT FOUND in database', [
                    'path' => $request->path(),
                    'method' => $request->method(),
                    'bearer_length' => $bearer ? strlen($bearer) : 0,
                    'bearer_prefix' => $bearer ? substr($bearer, 0, 20) . '...' : null,
                    'hashed_prefix' => $hashed ? substr($hashed, 0, 20) . '...' : null,
                    'recent_tokens_count' => $allTokens->count(),
                    'recent_tokens' => $allTokens->map(function($t) {
                        return [
                            'id' => $t->id,
                            'tokenable_id' => $t->tokenable_id,
                            'token_prefix' => substr($t->token, 0, 20) . '...',
                            'created_at' => $t->created_at,
                        ];
                    })->toArray(),
                ]);
            }
        }

        Log::info('Sanctum debug middleware', [
            'path' => $request->path(),
            'method' => $request->method(),
            'has_bearer' => $bearer !== null,
            'bearer_length' => $bearer ? strlen($bearer) : 0,
            'bearer_prefix' => $bearer ? substr($bearer, 0, 20) . '...' : null,
            'hashed_prefix' => $hashed ? substr($hashed, 0, 20) . '...' : null,
            'token_found' => $token !== null,
            'token_id' => $token?->id,
            'token_user_id' => $token?->tokenable_id,
            'abilities' => $token?->abilities,
            'expires_at' => $token?->expires_at?->toDateTimeString(),
            'is_expired' => $token?->expires_at ? $token->expires_at->isPast() : false,
        ]);

        return $next($request);
    }
}
