<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Annonce extends Model
{
    protected $fillable = [
        'user_id',
        'titre',
        'description',
        'type_bien_id',
        'statut',
        'ville',
        'quartier',
        'adresse',
        'prix',
        'superficie',
        'nombre_pieces',
        'nombre_chambres',
        'nombre_salles_bain',
        'equipements',
        'statut_annonce',
        'situation_geographique',
        'localisation_exacte',
        'superficie_terrain',
        'type_terrain',
        'statut_approbation',
        'titre_propriete',
        'nombre_vues',
        'nombre_favoris',
        'nombre_contacts',
    ];

    protected $casts = [
        'equipements' => 'array',
        'prix' => 'decimal:2',
        'superficie' => 'decimal:2',
        'superficie_terrain' => 'decimal:2',
        'nombre_vues' => 'integer',
        'nombre_favoris' => 'integer',
        'nombre_contacts' => 'integer',
    ];

    /**
     * Relation avec l'utilisateur propriétaire
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Relation avec les photos
     */
    public function photos(): HasMany
    {
        return $this->hasMany(Photo::class)->orderBy('ordre');
    }

    public function typeBien(): BelongsTo
    {
        return $this->belongsTo(TypeBien::class);
    }

    /**
     * Relation avec les favoris
     */
    public function favoris(): HasMany
    {
        return $this->hasMany(Favori::class);
    }

    /**
     * Relation avec les messages
     */
    public function messages(): HasMany
    {
        return $this->hasMany(Message::class);
    }

    /**
     * Obtenir la photo principale
     */
    public function photoPrincipale()
    {
        return $this->photos()->where('est_principale', true)->first();
    }

    /**
     * Scope pour les annonces actives
     */
    public function scopeActive($query)
    {
        return $query->where('statut_annonce', 'active');
    }

    /**
     * Scope pour filtrer par type de bien
     */
    public function scopeByTypeBien($query, $type)
    {
        return $query->where('type_bien', $type);
    }

    /**
     * Scope pour filtrer par statut (vente/location)
     */
    public function scopeByStatut($query, $statut)
    {
        return $query->where('statut', $statut);
    }

    /**
     * Scope pour filtrer par ville
     */
    public function scopeByVille($query, $ville)
    {
        return $query->where('ville', 'like', "%{$ville}%");
    }

    /**
     * Scope pour filtrer par prix
     */
    public function scopeByPrix($query, $prixMin, $prixMax)
    {
        return $query->whereBetween('prix', [$prixMin, $prixMax]);
    }
}
