<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Favori;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class FavoriController extends Controller
{
    /**
     * Ajouter une annonce aux favoris
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'annonce_id' => 'required|integer|exists:annonces,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Erreur de validation',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = $request->user();

        // Vérifier si le favori existe déjà
        $existingFavori = Favori::where('user_id', $user->id)
            ->where('annonce_id', $request->annonce_id)
            ->first();

        if ($existingFavori) {
            return response()->json([
                'success' => true,
                'message' => 'Annonce déjà dans les favoris',
                'data' => $existingFavori->load(['annonce', 'user']),
            ], 200);
    }

        // Créer le favori
        $favori = Favori::create([
            'user_id' => $user->id,
            'annonce_id' => $request->annonce_id,
        ]);

        $favori->load(['annonce.photos', 'annonce.user']);

        return response()->json([
            'success' => true,
            'message' => 'Annonce ajoutée aux favoris',
            'data' => $favori,
        ], 201);
    }

    /**
     * Retirer une annonce des favoris
     */
    public function destroy($id)
    {
        $favori = Favori::find($id);

        if (!$favori) {
            return response()->json([
                'success' => false,
                'message' => 'Favori non trouvé',
            ], 404);
    }

        // Vérifier que l'utilisateur est propriétaire du favori
        if ($favori->user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Non autorisé',
            ], 403);
        }

        $favori->delete();

        return response()->json([
            'success' => true,
            'message' => 'Annonce retirée des favoris',
        ], 200);
    }

    /**
     * Obtenir les favoris de l'utilisateur connecté
     */
    public function myFavoris(Request $request)
    {
        $user = $request->user();
        $perPage = $request->input('per_page', 15);
        $page = $request->input('page', 1);

        $favoris = Favori::where('user_id', $user->id)
            ->with(['annonce.photos', 'annonce.user'])
            ->orderBy('created_at', 'desc')
            ->paginate($perPage, ['*'], 'page', $page);

        return response()->json([
            'success' => true,
            'data' => $favoris->items(),
            'pagination' => [
                'current_page' => $favoris->currentPage(),
                'last_page' => $favoris->lastPage(),
                'per_page' => $favoris->perPage(),
                'total' => $favoris->total(),
            ],
        ], 200);
    }

    /**
     * Vérifier si une annonce est dans les favoris de l'utilisateur
     */
    public function checkFavori(Request $request, $annonceId)
    {
        $user = $request->user();

        $favori = Favori::where('user_id', $user->id)
            ->where('annonce_id', $annonceId)
            ->first();

        return response()->json([
            'success' => true,
            'is_favori' => $favori !== null,
            'favori_id' => $favori?->id,
        ], 200);
    }
}
