<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class UserController extends Controller
{
    /**
     * Obtenir le profil de l'utilisateur connecté
     */
    public function profile(Request $request)
    {
        $user = $request->user();
        $user->load('role');
        
        return response()->json([
            'success' => true,
            'data' => $user,
        ]);
    }

    /**
     * Mettre à jour le profil de l'utilisateur
     */
    public function updateProfile(Request $request)
    {
        $user = $request->user();

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|required|string|max:255',
            'email' => 'sometimes|required|string|email|max:255|unique:users,email,' . $user->id,
            'numero' => 'sometimes|required|string|max:20|unique:users,numero,' . $user->id,
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Erreur de validation',
                'errors' => $validator->errors()
            ], 422);
        }

        $user->update($request->only(['name', 'email', 'numero']));
        $user->load('role');

        return response()->json([
            'success' => true,
            'message' => 'Profil mis à jour avec succès',
            'data' => $user,
        ]);
    }

    /**
     * Mettre à jour la photo de profil
     */
    public function updateProfilePhoto(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'photo' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Erreur de validation',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = $request->user();

        try {
            // Supprimer l'ancienne photo si elle existe
            if ($user->profile_photo) {
                $oldPhotoPath = str_replace('/storage/', '', $user->profile_photo);
                if (Storage::disk('public')->exists($oldPhotoPath)) {
                    Storage::disk('public')->delete($oldPhotoPath);
                }
            }

            // Stocker la nouvelle photo
            $path = $request->file('photo')->store('profile-photos', 'public');
            
            // Mettre à jour le champ profile_photo avec l'URL complète
            $user->profile_photo = '/storage/' . $path;
            $user->save();

            Log::info('Profile photo updated', [
                'user_id' => $user->id,
                'path' => $user->profile_photo,
            ]);

            $user->load('role');

            return response()->json([
                'success' => true,
                'message' => 'Photo de profil mise à jour avec succès',
                'data' => $user,
            ]);
        } catch (\Exception $e) {
            Log::error('Error updating profile photo', [
                'user_id' => $user->id,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Erreur lors de la mise à jour de la photo',
            ], 500);
        }
    }
}
